'use client';
import React, { createContext, useContext, useState, useEffect } from 'react';

interface Tenant {
  id: string;
  name: string;
  room: string;
  hasPaid: boolean;
  amount: number;
}

interface Due {
  id: string;
  month: string;
  tenants: Tenant[];
}

interface Expense {
  id: string;
  date: string;
  description: string;
  amount: number;
  category: string;
}

interface AppContextType {
  dues: Due[];
  expenses: Expense[];
  addDue: (due: Due) => void;
  addExpense: (expense: Expense) => void;
  updateDue: (due: Due) => void;
  updateExpense: (expense: Expense) => void;
  deleteDue: (id: string) => void;
  deleteExpense: (id: string) => void;
  getTotalIncome: () => number;
  getTotalExpenses: () => number;
  getBalance: () => number;
}

const AppContext = createContext<AppContextType | undefined>(undefined);

export function AppProvider({ children }: { children: React.ReactNode }) {
  const [dues, setDues] = useState<Due[]>([]);
  const [expenses, setExpenses] = useState<Expense[]>([]);

  // Load data from localStorage on initial render
  useEffect(() => {
    const savedDues = localStorage.getItem('dues');
    const savedExpenses = localStorage.getItem('expenses');
    
    if (savedDues) {
      setDues(JSON.parse(savedDues));
    }
    
    if (savedExpenses) {
      setExpenses(JSON.parse(savedExpenses));
    }
  }, []);

  // Save data to localStorage whenever it changes
  useEffect(() => {
    localStorage.setItem('dues', JSON.stringify(dues));
  }, [dues]);

  useEffect(() => {
    localStorage.setItem('expenses', JSON.stringify(expenses));
  }, [expenses]);

  const addDue = (due: Due) => {
    setDues([...dues, due]);
  };

  const addExpense = (expense: Expense) => {
    setExpenses([...expenses, expense]);
  };

  const updateDue = (updatedDue: Due) => {
    setDues(dues.map(due => due.id === updatedDue.id ? updatedDue : due));
  };

  const updateExpense = (updatedExpense: Expense) => {
    setExpenses(expenses.map(expense => expense.id === updatedExpense.id ? updatedExpense : expense));
  };

  const deleteDue = (id: string) => {
    setDues(dues.filter(due => due.id !== id));
  };

  const deleteExpense = (id: string) => {
    setExpenses(expenses.filter(expense => expense.id !== id));
  };

  const getTotalIncome = () => {
    return dues.reduce((total, due) => {
      return total + due.tenants.reduce((sum, tenant) => sum + tenant.amount, 0);
    }, 0);
  };

  const getTotalExpenses = () => {
    return expenses.reduce((total, expense) => total + expense.amount, 0);
  };

  const getBalance = () => {
    return getTotalIncome() - getTotalExpenses();
  };

  return (
    <AppContext.Provider value={{
      dues,
      expenses,
      addDue,
      addExpense,
      updateDue,
      updateExpense,
      deleteDue,
      deleteExpense,
      getTotalIncome,
      getTotalExpenses,
      getBalance
    }}>
      {children}
    </AppContext.Provider>
  );
}

export function useApp() {
  const context = useContext(AppContext);
  if (context === undefined) {
    throw new Error('useApp must be used within an AppProvider');
  }
  return context;
} 