'use client';
import React, { useState } from 'react';
import Link from 'next/link';
import { useApp } from '../context/AppContext';

interface Tenant {
  id: string;
  name: string;
  room: string;
  hasPaid: boolean;
  amount: number;
  isEditing?: boolean;
}

interface Due {
  id: string;
  month: string;
  tenants: Tenant[];
}

// Jumlah iuran tetap untuk setiap penghuni
const FIXED_DUES: Record<string, number> = {
  '1': 150000, // Dito
  '2': 150000, // Satria
  '3': 180000, // Nabil
  '4': 120000, // Izas
};

const INITIAL_TENANTS: Tenant[] = [
  { id: '1', name: 'Dito', room: '1', hasPaid: false, amount: FIXED_DUES['1'] },
  { id: '2', name: 'Satria', room: '2', hasPaid: false, amount: FIXED_DUES['2'] },
  { id: '3', name: 'Nabil', room: '3', hasPaid: false, amount: FIXED_DUES['3'] },
  { id: '4', name: 'Izas', room: '4', hasPaid: false, amount: FIXED_DUES['4'] }
];

export default function DuesPage() {
  const { dues, addDue, updateDue } = useApp();
  const [newDue, setNewDue] = useState({
    month: '',
    tenants: INITIAL_TENANTS
  });
  const [editingDueId, setEditingDueId] = useState<string | null>(null);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    const due: Due = {
      id: Date.now().toString(),
      month: newDue.month,
      tenants: newDue.tenants
    };
    addDue(due);
    setNewDue({
      month: '',
      tenants: INITIAL_TENANTS
    });
  };

  const togglePayment = (dueId: string, tenantId: string) => {
    const due = dues.find(d => d.id === dueId);
    if (due) {
      const updatedDue = {
        ...due,
        tenants: due.tenants.map(tenant => {
          if (tenant.id === tenantId) {
            return { ...tenant, hasPaid: !tenant.hasPaid };
          }
          return tenant;
        })
      };
      updateDue(updatedDue);
    }
  };

  const startEditing = (dueId: string) => {
    setEditingDueId(dueId);
  };

  const saveEdit = (dueId: string) => {
    const due = dues.find(d => d.id === dueId);
    if (due) {
      const updatedDue = {
        ...due,
        tenants: due.tenants.map(tenant => ({
          ...tenant,
          isEditing: false
        }))
      };
      updateDue(updatedDue);
      setEditingDueId(null);
    }
  };

  const cancelEdit = (dueId: string) => {
    const due = dues.find(d => d.id === dueId);
    if (due) {
      const updatedDue = {
        ...due,
        tenants: due.tenants.map(tenant => ({
          ...tenant,
          amount: FIXED_DUES[tenant.id],
          isEditing: false
        }))
      };
      updateDue(updatedDue);
      setEditingDueId(null);
    }
  };

  const handleAmountChange = (dueId: string, tenantId: string, value: string) => {
    const due = dues.find(d => d.id === dueId);
    if (due) {
      const updatedDue = {
        ...due,
        tenants: due.tenants.map(tenant => {
          if (tenant.id === tenantId) {
            return { ...tenant, amount: Number(value) || 0 };
          }
          return tenant;
        })
      };
      updateDue(updatedDue);
    }
  };

  const getTotalAmount = (tenants: Tenant[]) => {
    return tenants.reduce((sum, tenant) => sum + tenant.amount, 0);
  };

  const getPaidAmount = (tenants: Tenant[]) => {
    return tenants.reduce((sum, tenant) => {
      return sum + (tenant.hasPaid ? tenant.amount : 0);
    }, 0);
  };

  const getUnpaidAmount = (tenants: Tenant[]) => {
    return tenants.reduce((sum, tenant) => {
      return sum + (!tenant.hasPaid ? tenant.amount : 0);
    }, 0);
  };

  return (
    <div className="py-6">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Iuran Kontrakan</h1>
          <Link href="/" className="text-blue-600 hover:text-blue-800 flex items-center">
            <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 mr-1" viewBox="0 0 20 20" fill="currentColor">
              <path fillRule="evenodd" d="M9.707 16.707a1 1 0 01-1.414 0l-6-6a1 1 0 010-1.414l6-6a1 1 0 011.414 1.414L5.414 9H17a1 1 0 110 2H5.414l4.293 4.293a1 1 0 010 1.414z" clipRule="evenodd" />
            </svg>
            Kembali
          </Link>
        </div>
        
        <div className="bg-white shadow-sm rounded-lg overflow-hidden mb-8">
          <div className="p-6">
            <h2 className="text-lg font-medium text-gray-900 mb-4">Tambah Iuran Baru</h2>
            <form onSubmit={handleSubmit}>
              <div className="mb-6">
                <label className="block text-sm font-medium text-gray-700">Bulan</label>
                <select
                  value={newDue.month}
                  onChange={(e) => setNewDue({ ...newDue, month: e.target.value })}
                  className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                  required
                >
                  <option value="">Pilih Bulan</option>
                  <option value="Januari">Januari</option>
                  <option value="Februari">Februari</option>
                  <option value="Maret">Maret</option>
                  <option value="April">April</option>
                  <option value="Mei">Mei</option>
                  <option value="Juni">Juni</option>
                  <option value="Juli">Juli</option>
                  <option value="Agustus">Agustus</option>
                  <option value="September">September</option>
                  <option value="Oktober">Oktober</option>
                  <option value="November">November</option>
                  <option value="Desember">Desember</option>
                </select>
              </div>

              <div className="mb-6">
                <h3 className="text-md font-medium text-gray-900 mb-3">Jumlah Iuran per Penghuni</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  {newDue.tenants.map((tenant) => (
                    <div key={tenant.id} className="p-4 border rounded-lg">
                      <div className="flex justify-between items-center mb-2">
                        <span className="font-medium text-gray-700">{tenant.name} (Kamar {tenant.room})</span>
                      </div>
                      <div>
                        <label className="block text-sm text-gray-500">Jumlah Iuran (Rp)</label>
                        <input
                          type="number"
                          value={tenant.amount}
                          readOnly
                          className="mt-1 block w-full rounded-md border-gray-300 bg-gray-50 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        />
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              <div className="flex justify-between items-center">
                <div className="text-lg font-medium text-gray-900">
                  Total Iuran: <span className="text-blue-600">Rp {getTotalAmount(newDue.tenants).toLocaleString()}</span>
                </div>
                <button
                  type="submit"
                  className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                >
                  Tambah Iuran
                </button>
              </div>
            </form>
          </div>
        </div>

        <div className="bg-white shadow-sm rounded-lg overflow-hidden">
          <div className="px-4 py-5 sm:px-6 border-b border-gray-200">
            <h3 className="text-lg leading-6 font-medium text-gray-900">Daftar Iuran</h3>
          </div>
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Periode</th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Dito (Kamar 1)</th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Satria (Kamar 2)</th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nabil (Kamar 3)</th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Izas (Kamar 4)</th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total</th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Aksi</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {dues.length === 0 ? (
                  <tr>
                    <td colSpan={8} className="px-6 py-4 text-center text-sm text-gray-500">
                      Belum ada data iuran
                    </td>
                  </tr>
                ) : (
                  dues.map((due) => (
                    <tr key={due.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">{due.month}</td>
                      {due.tenants.map((tenant) => (
                        <td key={tenant.id} className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                          <div className="flex flex-col">
                            {editingDueId === due.id ? (
                              <input
                                type="number"
                                value={tenant.amount}
                                onChange={(e) => handleAmountChange(due.id, tenant.id, e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                              />
                            ) : (
                              <span className="font-medium">Rp {tenant.amount.toLocaleString()}</span>
                            )}
                            <div className="flex items-center mt-1">
                              <input
                                type="checkbox"
                                checked={tenant.hasPaid}
                                onChange={() => togglePayment(due.id, tenant.id)}
                                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                              />
                              <span className="ml-2 text-xs">
                                {tenant.hasPaid ? 'Sudah Bayar' : 'Belum Bayar'}
                              </span>
                            </div>
                          </div>
                        </td>
                      ))}
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                        <div className="flex flex-col">
                          <span>Total: Rp {getTotalAmount(due.tenants).toLocaleString()}</span>
                          <span className="text-green-600">Sudah Bayar: Rp {getPaidAmount(due.tenants).toLocaleString()}</span>
                          <span className="text-red-600">Belum Bayar: Rp {getUnpaidAmount(due.tenants).toLocaleString()}</span>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm">
                        {due.tenants.every(tenant => tenant.hasPaid) ? (
                          <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">
                            Lunas
                          </span>
                        ) : (
                          <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800">
                            Belum Lunas
                          </span>
                        )}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm">
                        {editingDueId === due.id ? (
                          <div className="flex space-x-2">
                            <button
                              onClick={() => saveEdit(due.id)}
                              className="text-green-600 hover:text-green-900"
                            >
                              Simpan
                            </button>
                            <button
                              onClick={() => cancelEdit(due.id)}
                              className="text-red-600 hover:text-red-900"
                            >
                              Batal
                            </button>
                          </div>
                        ) : (
                          <button
                            onClick={() => startEditing(due.id)}
                            className="text-blue-600 hover:text-blue-900"
                          >
                            Edit
                          </button>
                        )}
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </div>
  );
} 